"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getNullActionPipelineManager = getNullActionPipelineManager;
exports.default = exports.ActionPipeline = void 0;
function partial(fn, ...args) {
  return function wrapped() {
    return fn(...args);
  };
}
class ActionPipeline {
  constructor(actionKey) {
    this.actionKey = actionKey;
    this.middleware = [];
  }
  run(fn, repository, ...args) {
    const pipelineFn = this.middleware.map(middleware => middleware.fn).reduceRight((acc, nextFn) => partial(nextFn, acc, repository, ...args), partial(fn, ...args));
    return pipelineFn(...args);
  }
  addMiddleware(name, fn) {
    if (!name) {
      throw new Error('Middleware must be registered with a unique middleware name');
    }
    if (this.middleware.find(middleware => middleware.name === name)) {
      throw new Error(`A middleware with the name ${name} is already registered`);
    }
    this.middleware.push({
      name,
      fn
    });
  }
}
exports.ActionPipeline = ActionPipeline;
class ActionPipelineManager {
  constructor({
    actionNames
  } = {
    actionNames: []
  }) {
    this.pipelines = new Map();
    this.actionKeys = {};
    actionNames.forEach(actionName => {
      const actionKey = Symbol(actionName);
      this.actionKeys[actionName] = actionKey;
      this.pipelines.set(actionKey, new ActionPipeline(actionKey));
    });
  }
  getPipeline(actionKey) {
    const pipeline = this.pipelines.get(actionKey);
    if (actionKey && pipeline) {
      return pipeline;
    } else {
      throw new Error(`${actionKey} is not a known action`);
    }
  }
}
exports.default = ActionPipelineManager;
class NullActionPipeline extends ActionPipeline {
  run(fn, repository, ...args) {
    return fn(...args);
  }
  addMiddleware() {
    throw new Error('Cannot add middleware to a null pipeline');
  }
}
const nullPipeline = new NullActionPipeline();
class NullActionPipelineManager extends ActionPipelineManager {
  getPipeline() {
    return nullPipeline;
  }
}
const nullActionPipelineManager = new NullActionPipelineManager();
function getNullActionPipelineManager() {
  return nullActionPipelineManager;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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