"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
class Task {
  constructor(fn, parallel = true) {
    this.fn = fn;
    this.parallel = parallel;
    this.promise = new Promise((resolve, reject) => {
      this.resolve = resolve;
      this.reject = reject;
    });
  }
  async execute() {
    try {
      const value = await this.fn.call(undefined);
      this.resolve(value);
    } catch (err) {
      this.reject(err);
    }
  }
  runsInParallel() {
    return this.parallel;
  }
  runsInSerial() {
    return !this.parallel;
  }
  getPromise() {
    return this.promise;
  }
}
class AsyncQueue {
  constructor(options = {}) {
    this.parallelism = options.parallelism || 1;
    this.nonParallelizableOperation = false;
    this.tasksInProgress = 0;
    this.queue = [];
  }
  push(fn, {
    parallel
  } = {
    parallel: true
  }) {
    const task = new Task(fn, parallel);
    this.queue.push(task);
    this.processQueue();
    return task.getPromise();
  }
  processQueue() {
    if (!this.queue.length || this.nonParallelizableOperation || this.disposed) {
      return;
    }
    const task = this.queue[0];
    const canRunParallelOp = task.runsInParallel() && this.tasksInProgress < this.parallelism;
    const canRunSerialOp = task.runsInSerial() && this.tasksInProgress === 0;
    if (canRunSerialOp || canRunParallelOp) {
      this.processTask(task, task.runsInParallel());
      this.queue.shift();
      this.processQueue();
    }
  }
  async processTask(task, runsInParallel) {
    if (this.disposed) {
      return;
    }
    this.tasksInProgress++;
    if (!runsInParallel) {
      this.nonParallelizableOperation = true;
    }
    try {
      await task.execute();
    } finally {
      this.tasksInProgress--;
      this.nonParallelizableOperation = false;
      this.processQueue();
    }
  }
  dispose() {
    this.queue = [];
    this.disposed = true;
  }
}
exports.default = AsyncQueue;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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