"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
var _net = _interopRequireDefault(require("net"));
var _eventKit = require("event-kit");
var _helpers = require("./helpers");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
class GitPromptServer {
  constructor(gitTempDir) {
    this.emitter = new _eventKit.Emitter();
    this.gitTempDir = gitTempDir;
    this.address = null;
  }
  async start(promptForInput) {
    this.promptForInput = promptForInput;
    await this.gitTempDir.ensure();
    this.server = await this.startListening(this.gitTempDir.getSocketOptions());
  }
  getAddress() {
    /* istanbul ignore if */
    if (!this.address) {
      throw new Error('Server is not listening');
    } else if (this.address.port) {
      // TCP socket
      return `tcp:${this.address.port}`;
    } else {
      // Unix domain socket
      return `unix:${(0, _helpers.normalizeGitHelperPath)(this.address)}`;
    }
  }
  startListening(socketOptions) {
    return new Promise(resolve => {
      const server = _net.default.createServer({
        allowHalfOpen: true
      }, connection => {
        connection.setEncoding('utf8');
        let payload = '';
        connection.on('data', data => {
          payload += data;
        });
        connection.on('end', () => {
          this.handleData(connection, payload);
        });
      });
      server.listen(socketOptions, () => {
        this.address = server.address();
        resolve(server);
      });
    });
  }
  async handleData(connection, data) {
    let query;
    try {
      query = JSON.parse(data);
      const answer = await this.promptForInput(query);
      await new Promise(resolve => {
        connection.end(JSON.stringify(answer), 'utf8', resolve);
      });
    } catch (e) {
      this.emitter.emit('did-cancel', query.pid ? {
        handlerPid: query.pid
      } : undefined);
    }
  }
  onDidCancel(cb) {
    return this.emitter.on('did-cancel', cb);
  }
  async terminate() {
    await new Promise(resolve => this.server.close(resolve));
    this.emitter.dispose();
  }
}
exports.default = GitPromptServer;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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