"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.expectRelayQuery = expectRelayQuery;
exports.clearRelayExpectations = clearRelayExpectations;
exports.default = void 0;
var _util = _interopRequireDefault(require("util"));
var _relayRuntime = require("relay-runtime");
var _moment = _interopRequireDefault(require("moment"));
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
const LODASH_ISEQUAL = 'lodash.isequal';
let isEqual = null;
const relayEnvironmentPerURL = new Map();
const tokenPerURL = new Map();
const fetchPerURL = new Map();
const responsesByQuery = new Map();
function logRatelimitApi(headers) {
  const remaining = headers.get('x-ratelimit-remaining');
  const total = headers.get('x-ratelimit-limit');
  const resets = headers.get('x-ratelimit-reset');
  const resetsIn = _moment.default.unix(parseInt(resets, 10)).from();

  // eslint-disable-next-line no-console
  console.debug(`GitHub API Rate Limiting Info: ${remaining}/${total} requests left — resets ${resetsIn}`);
}
function expectRelayQuery(operationPattern, response) {
  let resolve, reject;
  const handler = typeof response === 'function' ? response : () => ({
    data: response
  });
  const promise = new Promise((resolve0, reject0) => {
    resolve = resolve0;
    reject = reject0;
  });
  const existing = responsesByQuery.get(operationPattern.name) || [];
  existing.push({
    promise,
    handler,
    variables: operationPattern.variables || {},
    trace: operationPattern.trace
  });
  responsesByQuery.set(operationPattern.name, existing);
  const disable = () => responsesByQuery.delete(operationPattern.name);
  return {
    promise,
    resolve,
    reject,
    disable
  };
}
function clearRelayExpectations() {
  responsesByQuery.clear();
  relayEnvironmentPerURL.clear();
  tokenPerURL.clear();
  fetchPerURL.clear();
  responsesByQuery.clear();
}
function createFetchQuery(url) {
  if (atom.inSpecMode()) {
    return function specFetchQuery(operation, variables, _cacheConfig, _uploadables) {
      const expectations = responsesByQuery.get(operation.name) || [];
      const match = expectations.find(expectation => {
        if (isEqual === null) {
          // Lazily require lodash.isequal so we can keep it as a dev dependency.
          // Require indirectly to trick electron-link into not following this.
          isEqual = require(LODASH_ISEQUAL);
        }
        return isEqual(expectation.variables, variables);
      });
      if (!match) {
        // eslint-disable-next-line no-console
        console.log(`GraphQL query ${operation.name} was:\n  ${operation.text.replace(/\n/g, '\n  ')}\n` + _util.default.inspect(variables));
        const e = new Error(`Unexpected GraphQL query: ${operation.name}`);
        e.rawStack = e.stack;
        throw e;
      }
      const responsePromise = match.promise.then(() => {
        return match.handler(operation);
      });
      if (match.trace) {
        // eslint-disable-next-line no-console
        console.log(`[Relay] query "${operation.name}":\n${operation.text}`);
        responsePromise.then(result => {
          // eslint-disable-next-line no-console
          console.log(`[Relay] response "${operation.name}":`, result);
        }, err => {
          // eslint-disable-next-line no-console
          console.error(`[Relay] error "${operation.name}":\n${err.stack || err}`);
          throw err;
        });
      }
      return responsePromise;
    };
  }
  return async function fetchQuery(operation, variables, _cacheConfig, _uploadables) {
    const currentToken = tokenPerURL.get(url);
    let response;
    try {
      response = await fetch(url, {
        method: 'POST',
        headers: {
          'content-type': 'application/json',
          'Authorization': `bearer ${currentToken}`,
          'Accept': 'application/vnd.github.antiope-preview+json'
        },
        body: JSON.stringify({
          query: operation.text,
          variables
        })
      });
    } catch (e) {
      // A network error was encountered. Mark it so that QueryErrorView and ErrorView can distinguish these, because
      // the errors from "fetch" are TypeErrors without much information.
      e.network = true;
      e.rawStack = e.stack;
      throw e;
    }
    try {
      atom && atom.inDevMode() && logRatelimitApi(response.headers);
    } catch (_e) {/* do nothing */}
    if (response.status !== 200) {
      const e = new Error(`GraphQL API endpoint at ${url} returned ${response.status}`);
      e.response = response;
      e.responseText = await response.text();
      e.rawStack = e.stack;
      throw e;
    }
    const payload = await response.json();
    if (payload && payload.errors && payload.errors.length > 0) {
      const e = new Error(`GraphQL API endpoint at ${url} returned an error for query ${operation.name}.`);
      e.response = response;
      e.errors = payload.errors;
      e.rawStack = e.stack;
      throw e;
    }
    return payload;
  };
}
class RelayNetworkLayerManager {
  static getEnvironmentForHost(endpoint, token) {
    const url = endpoint.getGraphQLRoot();
    let {
      environment,
      network
    } = relayEnvironmentPerURL.get(url) || {};
    tokenPerURL.set(url, token);
    if (!environment) {
      if (!token) {
        throw new Error(`You must authenticate to ${endpoint.getHost()} first.`);
      }
      const source = new _relayRuntime.RecordSource();
      const store = new _relayRuntime.Store(source);
      network = _relayRuntime.Network.create(this.getFetchQuery(endpoint, token));
      environment = new _relayRuntime.Environment({
        network,
        store
      });
      relayEnvironmentPerURL.set(url, {
        environment,
        network
      });
    }
    return environment;
  }
  static getFetchQuery(endpoint, token) {
    const url = endpoint.getGraphQLRoot();
    tokenPerURL.set(url, token);
    let fetch = fetchPerURL.get(url);
    if (!fetch) {
      fetch = createFetchQuery(url);
      fetchPerURL.set(fetch);
    }
    return fetch;
  }
}
exports.default = RelayNetworkLayerManager;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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