"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.incrementCounter = incrementCounter;
exports.addTiming = addTiming;
exports.addEvent = addEvent;
exports.FakeReporter = exports.reporterProxy = exports.FIVE_MINUTES_IN_MILLISECONDS = void 0;
const pjson = require('../package.json');
const FIVE_MINUTES_IN_MILLISECONDS = 1000 * 60 * 5;

// this class allows us to call reporter methods
// before the reporter is actually loaded, since we don't want to
// assume that the metrics package will load before the GitHub package.
exports.FIVE_MINUTES_IN_MILLISECONDS = FIVE_MINUTES_IN_MILLISECONDS;
class ReporterProxy {
  constructor() {
    this.reporter = null;
    this.events = [];
    this.timings = [];
    this.counters = [];
    this.gitHubPackageVersion = pjson.version;
    this.timeout = null;
  }

  // function that is called after the reporter is actually loaded, to
  // set the reporter and send any data that have accumulated while it was loading.
  setReporter(reporter) {
    this.reporter = reporter;
    this.events.forEach(customEvent => {
      this.reporter.addCustomEvent(customEvent.eventType, customEvent.event);
    });
    this.events = [];
    this.timings.forEach(timing => {
      this.reporter.addTiming(timing.eventType, timing.durationInMilliseconds, timing.metadata);
    });
    this.timings = [];
    this.counters.forEach(counterName => {
      this.reporter.incrementCounter(counterName);
    });
    this.counters = [];
  }
  incrementCounter(counterName) {
    if (this.reporter === null) {
      this.startTimer();
      this.counters.push(counterName);
      return;
    }
    this.reporter.incrementCounter(counterName);
  }
  addTiming(eventType, durationInMilliseconds, metadata = {}) {
    if (this.reporter === null) {
      this.startTimer();
      this.timings.push({
        eventType,
        durationInMilliseconds,
        metadata
      });
      return;
    }
    this.reporter.addTiming(eventType, durationInMilliseconds, metadata);
  }
  addEvent(eventType, event) {
    if (this.reporter === null) {
      this.startTimer();
      this.events.push({
        eventType,
        event
      });
      return;
    }
    this.reporter.addCustomEvent(eventType, event);
  }
  startTimer() {
    if (this.timeout !== null) {
      return;
    }

    // if for some reason a user disables the metrics package, we don't want to
    // just keep accumulating events in memory until the heat death of the universe.
    // Use a no-op class, clear all queues, move on with our lives.
    this.timeout = setTimeout(FIVE_MINUTES_IN_MILLISECONDS, () => {
      if (this.reporter === null) {
        this.setReporter(new FakeReporter());
        this.events = [];
        this.timings = [];
        this.counters = [];
      }
    });
  }
}
const reporterProxy = new ReporterProxy();
exports.reporterProxy = reporterProxy;
class FakeReporter {
  addCustomEvent() {}
  addTiming() {}
  incrementCounter() {}
}
exports.FakeReporter = FakeReporter;
function incrementCounter(counterName) {
  reporterProxy.incrementCounter(counterName);
}
function addTiming(eventType, durationInMilliseconds, metadata = {}) {
  metadata.gitHubPackageVersion = reporterProxy.gitHubPackageVersion;
  reporterProxy.addTiming(eventType, durationInMilliseconds, metadata);
}
function addEvent(eventType, event) {
  event.gitHubPackageVersion = reporterProxy.gitHubPackageVersion;
  reporterProxy.addEvent(eventType, event);
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJuYW1lcyI6WyJwanNvbiIsInJlcXVpcmUiLCJGSVZFX01JTlVURVNfSU5fTUlMTElTRUNPTkRTIiwiZXhwb3J0cyIsIlJlcG9ydGVyUHJveHkiLCJjb25zdHJ1Y3RvciIsInJlcG9ydGVyIiwiZXZlbnRzIiwidGltaW5ncyIsImNvdW50ZXJzIiwiZ2l0SHViUGFja2FnZVZlcnNpb24iLCJ2ZXJzaW9uIiwidGltZW91dCIsInNldFJlcG9ydGVyIiwiZm9yRWFjaCIsImN1c3RvbUV2ZW50IiwiYWRkQ3VzdG9tRXZlbnQiLCJldmVudFR5cGUiLCJldmVudCIsInRpbWluZyIsImFkZFRpbWluZyIsImR1cmF0aW9uSW5NaWxsaXNlY29uZHMiLCJtZXRhZGF0YSIsImNvdW50ZXJOYW1lIiwiaW5jcmVtZW50Q291bnRlciIsInN0YXJ0VGltZXIiLCJwdXNoIiwiYWRkRXZlbnQiLCJzZXRUaW1lb3V0IiwiRmFrZVJlcG9ydGVyIiwicmVwb3J0ZXJQcm94eSJdLCJzb3VyY2VzIjpbInJlcG9ydGVyLXByb3h5LmpzIl0sInNvdXJjZXNDb250ZW50IjpbImNvbnN0IHBqc29uID0gcmVxdWlyZSgnLi4vcGFja2FnZS5qc29uJyk7XG5cbmV4cG9ydCBjb25zdCBGSVZFX01JTlVURVNfSU5fTUlMTElTRUNPTkRTID0gMTAwMCAqIDYwICogNTtcblxuLy8gdGhpcyBjbGFzcyBhbGxvd3MgdXMgdG8gY2FsbCByZXBvcnRlciBtZXRob2RzXG4vLyBiZWZvcmUgdGhlIHJlcG9ydGVyIGlzIGFjdHVhbGx5IGxvYWRlZCwgc2luY2Ugd2UgZG9uJ3Qgd2FudCB0b1xuLy8gYXNzdW1lIHRoYXQgdGhlIG1ldHJpY3MgcGFja2FnZSB3aWxsIGxvYWQgYmVmb3JlIHRoZSBHaXRIdWIgcGFja2FnZS5cbmNsYXNzIFJlcG9ydGVyUHJveHkge1xuICBjb25zdHJ1Y3RvcigpIHtcbiAgICB0aGlzLnJlcG9ydGVyID0gbnVsbDtcbiAgICB0aGlzLmV2ZW50cyA9IFtdO1xuICAgIHRoaXMudGltaW5ncyA9IFtdO1xuICAgIHRoaXMuY291bnRlcnMgPSBbXTtcbiAgICB0aGlzLmdpdEh1YlBhY2thZ2VWZXJzaW9uID0gcGpzb24udmVyc2lvbjtcblxuICAgIHRoaXMudGltZW91dCA9IG51bGw7XG4gIH1cblxuICAvLyBmdW5jdGlvbiB0aGF0IGlzIGNhbGxlZCBhZnRlciB0aGUgcmVwb3J0ZXIgaXMgYWN0dWFsbHkgbG9hZGVkLCB0b1xuICAvLyBzZXQgdGhlIHJlcG9ydGVyIGFuZCBzZW5kIGFueSBkYXRhIHRoYXQgaGF2ZSBhY2N1bXVsYXRlZCB3aGlsZSBpdCB3YXMgbG9hZGluZy5cbiAgc2V0UmVwb3J0ZXIocmVwb3J0ZXIpIHtcbiAgICB0aGlzLnJlcG9ydGVyID0gcmVwb3J0ZXI7XG5cbiAgICB0aGlzLmV2ZW50cy5mb3JFYWNoKGN1c3RvbUV2ZW50ID0+IHtcbiAgICAgIHRoaXMucmVwb3J0ZXIuYWRkQ3VzdG9tRXZlbnQoY3VzdG9tRXZlbnQuZXZlbnRUeXBlLCBjdXN0b21FdmVudC5ldmVudCk7XG4gICAgfSk7XG4gICAgdGhpcy5ldmVudHMgPSBbXTtcblxuICAgIHRoaXMudGltaW5ncy5mb3JFYWNoKHRpbWluZyA9PiB7XG4gICAgICB0aGlzLnJlcG9ydGVyLmFkZFRpbWluZyh0aW1pbmcuZXZlbnRUeXBlLCB0aW1pbmcuZHVyYXRpb25Jbk1pbGxpc2Vjb25kcywgdGltaW5nLm1ldGFkYXRhKTtcbiAgICB9KTtcbiAgICB0aGlzLnRpbWluZ3MgPSBbXTtcblxuICAgIHRoaXMuY291bnRlcnMuZm9yRWFjaChjb3VudGVyTmFtZSA9PiB7XG4gICAgICB0aGlzLnJlcG9ydGVyLmluY3JlbWVudENvdW50ZXIoY291bnRlck5hbWUpO1xuICAgIH0pO1xuICAgIHRoaXMuY291bnRlcnMgPSBbXTtcbiAgfVxuXG4gIGluY3JlbWVudENvdW50ZXIoY291bnRlck5hbWUpIHtcbiAgICBpZiAodGhpcy5yZXBvcnRlciA9PT0gbnVsbCkge1xuICAgICAgdGhpcy5zdGFydFRpbWVyKCk7XG4gICAgICB0aGlzLmNvdW50ZXJzLnB1c2goY291bnRlck5hbWUpO1xuICAgICAgcmV0dXJuO1xuICAgIH1cblxuICAgIHRoaXMucmVwb3J0ZXIuaW5jcmVtZW50Q291bnRlcihjb3VudGVyTmFtZSk7XG4gIH1cblxuICBhZGRUaW1pbmcoZXZlbnRUeXBlLCBkdXJhdGlvbkluTWlsbGlzZWNvbmRzLCBtZXRhZGF0YSA9IHt9KSB7XG4gICAgaWYgKHRoaXMucmVwb3J0ZXIgPT09IG51bGwpIHtcbiAgICAgIHRoaXMuc3RhcnRUaW1lcigpO1xuICAgICAgdGhpcy50aW1pbmdzLnB1c2goe2V2ZW50VHlwZSwgZHVyYXRpb25Jbk1pbGxpc2Vjb25kcywgbWV0YWRhdGF9KTtcbiAgICAgIHJldHVybjtcbiAgICB9XG5cbiAgICB0aGlzLnJlcG9ydGVyLmFkZFRpbWluZyhldmVudFR5cGUsIGR1cmF0aW9uSW5NaWxsaXNlY29uZHMsIG1ldGFkYXRhKTtcbiAgfVxuXG4gIGFkZEV2ZW50KGV2ZW50VHlwZSwgZXZlbnQpIHtcbiAgICBpZiAodGhpcy5yZXBvcnRlciA9PT0gbnVsbCkge1xuICAgICAgdGhpcy5zdGFydFRpbWVyKCk7XG4gICAgICB0aGlzLmV2ZW50cy5wdXNoKHtldmVudFR5cGUsIGV2ZW50fSk7XG4gICAgICByZXR1cm47XG4gICAgfVxuXG4gICAgdGhpcy5yZXBvcnRlci5hZGRDdXN0b21FdmVudChldmVudFR5cGUsIGV2ZW50KTtcbiAgfVxuXG4gIHN0YXJ0VGltZXIoKSB7XG4gICAgaWYgKHRoaXMudGltZW91dCAhPT0gbnVsbCkge1xuICAgICAgcmV0dXJuO1xuICAgIH1cblxuICAgIC8vIGlmIGZvciBzb21lIHJlYXNvbiBhIHVzZXIgZGlzYWJsZXMgdGhlIG1ldHJpY3MgcGFja2FnZSwgd2UgZG9uJ3Qgd2FudCB0b1xuICAgIC8vIGp1c3Qga2VlcCBhY2N1bXVsYXRpbmcgZXZlbnRzIGluIG1lbW9yeSB1bnRpbCB0aGUgaGVhdCBkZWF0aCBvZiB0aGUgdW5pdmVyc2UuXG4gICAgLy8gVXNlIGEgbm8tb3AgY2xhc3MsIGNsZWFyIGFsbCBxdWV1ZXMsIG1vdmUgb24gd2l0aCBvdXIgbGl2ZXMuXG4gICAgdGhpcy50aW1lb3V0ID0gc2V0VGltZW91dChGSVZFX01JTlVURVNfSU5fTUlMTElTRUNPTkRTLCAoKSA9PiB7XG4gICAgICBpZiAodGhpcy5yZXBvcnRlciA9PT0gbnVsbCkge1xuICAgICAgICB0aGlzLnNldFJlcG9ydGVyKG5ldyBGYWtlUmVwb3J0ZXIoKSk7XG4gICAgICAgIHRoaXMuZXZlbnRzID0gW107XG4gICAgICAgIHRoaXMudGltaW5ncyA9IFtdO1xuICAgICAgICB0aGlzLmNvdW50ZXJzID0gW107XG4gICAgICB9XG4gICAgfSk7XG4gIH1cbn1cblxuZXhwb3J0IGNvbnN0IHJlcG9ydGVyUHJveHkgPSBuZXcgUmVwb3J0ZXJQcm94eSgpO1xuXG5leHBvcnQgY2xhc3MgRmFrZVJlcG9ydGVyIHtcbiAgYWRkQ3VzdG9tRXZlbnQoKSB7fVxuXG4gIGFkZFRpbWluZygpIHt9XG5cbiAgaW5jcmVtZW50Q291bnRlcigpIHt9XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBpbmNyZW1lbnRDb3VudGVyKGNvdW50ZXJOYW1lKSB7XG4gIHJlcG9ydGVyUHJveHkuaW5jcmVtZW50Q291bnRlcihjb3VudGVyTmFtZSk7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBhZGRUaW1pbmcoZXZlbnRUeXBlLCBkdXJhdGlvbkluTWlsbGlzZWNvbmRzLCBtZXRhZGF0YSA9IHt9KSB7XG4gIG1ldGFkYXRhLmdpdEh1YlBhY2thZ2VWZXJzaW9uID0gcmVwb3J0ZXJQcm94eS5naXRIdWJQYWNrYWdlVmVyc2lvbjtcbiAgcmVwb3J0ZXJQcm94eS5hZGRUaW1pbmcoZXZlbnRUeXBlLCBkdXJhdGlvbkluTWlsbGlzZWNvbmRzLCBtZXRhZGF0YSk7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBhZGRFdmVudChldmVudFR5cGUsIGV2ZW50KSB7XG4gIGV2ZW50LmdpdEh1YlBhY2thZ2VWZXJzaW9uID0gcmVwb3J0ZXJQcm94eS5naXRIdWJQYWNrYWdlVmVyc2lvbjtcbiAgcmVwb3J0ZXJQcm94eS5hZGRFdmVudChldmVudFR5cGUsIGV2ZW50KTtcbn1cbiJdLCJtYXBwaW5ncyI6Ijs7Ozs7Ozs7O0FBQUEsTUFBTUEsS0FBSyxHQUFHQyxPQUFPLENBQUMsaUJBQWlCLENBQUM7QUFFakMsTUFBTUMsNEJBQTRCLEdBQUcsSUFBSSxHQUFHLEVBQUUsR0FBRyxDQUFDOztBQUV6RDtBQUNBO0FBQ0E7QUFBQUMsT0FBQSxDQUFBRCw0QkFBQSxHQUFBQSw0QkFBQTtBQUNBLE1BQU1FLGFBQWEsQ0FBQztFQUNsQkMsV0FBV0EsQ0FBQSxFQUFHO0lBQ1osSUFBSSxDQUFDQyxRQUFRLEdBQUcsSUFBSTtJQUNwQixJQUFJLENBQUNDLE1BQU0sR0FBRyxFQUFFO0lBQ2hCLElBQUksQ0FBQ0MsT0FBTyxHQUFHLEVBQUU7SUFDakIsSUFBSSxDQUFDQyxRQUFRLEdBQUcsRUFBRTtJQUNsQixJQUFJLENBQUNDLG9CQUFvQixHQUFHVixLQUFLLENBQUNXLE9BQU87SUFFekMsSUFBSSxDQUFDQyxPQUFPLEdBQUcsSUFBSTtFQUNyQjs7RUFFQTtFQUNBO0VBQ0FDLFdBQVdBLENBQUNQLFFBQVEsRUFBRTtJQUNwQixJQUFJLENBQUNBLFFBQVEsR0FBR0EsUUFBUTtJQUV4QixJQUFJLENBQUNDLE1BQU0sQ0FBQ08sT0FBTyxDQUFDQyxXQUFXLElBQUk7TUFDakMsSUFBSSxDQUFDVCxRQUFRLENBQUNVLGNBQWMsQ0FBQ0QsV0FBVyxDQUFDRSxTQUFTLEVBQUVGLFdBQVcsQ0FBQ0csS0FBSyxDQUFDO0lBQ3hFLENBQUMsQ0FBQztJQUNGLElBQUksQ0FBQ1gsTUFBTSxHQUFHLEVBQUU7SUFFaEIsSUFBSSxDQUFDQyxPQUFPLENBQUNNLE9BQU8sQ0FBQ0ssTUFBTSxJQUFJO01BQzdCLElBQUksQ0FBQ2IsUUFBUSxDQUFDYyxTQUFTLENBQUNELE1BQU0sQ0FBQ0YsU0FBUyxFQUFFRSxNQUFNLENBQUNFLHNCQUFzQixFQUFFRixNQUFNLENBQUNHLFFBQVEsQ0FBQztJQUMzRixDQUFDLENBQUM7SUFDRixJQUFJLENBQUNkLE9BQU8sR0FBRyxFQUFFO0lBRWpCLElBQUksQ0FBQ0MsUUFBUSxDQUFDSyxPQUFPLENBQUNTLFdBQVcsSUFBSTtNQUNuQyxJQUFJLENBQUNqQixRQUFRLENBQUNrQixnQkFBZ0IsQ0FBQ0QsV0FBVyxDQUFDO0lBQzdDLENBQUMsQ0FBQztJQUNGLElBQUksQ0FBQ2QsUUFBUSxHQUFHLEVBQUU7RUFDcEI7RUFFQWUsZ0JBQWdCQSxDQUFDRCxXQUFXLEVBQUU7SUFDNUIsSUFBSSxJQUFJLENBQUNqQixRQUFRLEtBQUssSUFBSSxFQUFFO01BQzFCLElBQUksQ0FBQ21CLFVBQVUsQ0FBQyxDQUFDO01BQ2pCLElBQUksQ0FBQ2hCLFFBQVEsQ0FBQ2lCLElBQUksQ0FBQ0gsV0FBVyxDQUFDO01BQy9CO0lBQ0Y7SUFFQSxJQUFJLENBQUNqQixRQUFRLENBQUNrQixnQkFBZ0IsQ0FBQ0QsV0FBVyxDQUFDO0VBQzdDO0VBRUFILFNBQVNBLENBQUNILFNBQVMsRUFBRUksc0JBQXNCLEVBQUVDLFFBQVEsR0FBRyxDQUFDLENBQUMsRUFBRTtJQUMxRCxJQUFJLElBQUksQ0FBQ2hCLFFBQVEsS0FBSyxJQUFJLEVBQUU7TUFDMUIsSUFBSSxDQUFDbUIsVUFBVSxDQUFDLENBQUM7TUFDakIsSUFBSSxDQUFDakIsT0FBTyxDQUFDa0IsSUFBSSxDQUFDO1FBQUNULFNBQVM7UUFBRUksc0JBQXNCO1FBQUVDO01BQVEsQ0FBQyxDQUFDO01BQ2hFO0lBQ0Y7SUFFQSxJQUFJLENBQUNoQixRQUFRLENBQUNjLFNBQVMsQ0FBQ0gsU0FBUyxFQUFFSSxzQkFBc0IsRUFBRUMsUUFBUSxDQUFDO0VBQ3RFO0VBRUFLLFFBQVFBLENBQUNWLFNBQVMsRUFBRUMsS0FBSyxFQUFFO0lBQ3pCLElBQUksSUFBSSxDQUFDWixRQUFRLEtBQUssSUFBSSxFQUFFO01BQzFCLElBQUksQ0FBQ21CLFVBQVUsQ0FBQyxDQUFDO01BQ2pCLElBQUksQ0FBQ2xCLE1BQU0sQ0FBQ21CLElBQUksQ0FBQztRQUFDVCxTQUFTO1FBQUVDO01BQUssQ0FBQyxDQUFDO01BQ3BDO0lBQ0Y7SUFFQSxJQUFJLENBQUNaLFFBQVEsQ0FBQ1UsY0FBYyxDQUFDQyxTQUFTLEVBQUVDLEtBQUssQ0FBQztFQUNoRDtFQUVBTyxVQUFVQSxDQUFBLEVBQUc7SUFDWCxJQUFJLElBQUksQ0FBQ2IsT0FBTyxLQUFLLElBQUksRUFBRTtNQUN6QjtJQUNGOztJQUVBO0lBQ0E7SUFDQTtJQUNBLElBQUksQ0FBQ0EsT0FBTyxHQUFHZ0IsVUFBVSxDQUFDMUIsNEJBQTRCLEVBQUUsTUFBTTtNQUM1RCxJQUFJLElBQUksQ0FBQ0ksUUFBUSxLQUFLLElBQUksRUFBRTtRQUMxQixJQUFJLENBQUNPLFdBQVcsQ0FBQyxJQUFJZ0IsWUFBWSxDQUFDLENBQUMsQ0FBQztRQUNwQyxJQUFJLENBQUN0QixNQUFNLEdBQUcsRUFBRTtRQUNoQixJQUFJLENBQUNDLE9BQU8sR0FBRyxFQUFFO1FBQ2pCLElBQUksQ0FBQ0MsUUFBUSxHQUFHLEVBQUU7TUFDcEI7SUFDRixDQUFDLENBQUM7RUFDSjtBQUNGO0FBRU8sTUFBTXFCLGFBQWEsR0FBRyxJQUFJMUIsYUFBYSxDQUFDLENBQUM7QUFBQ0QsT0FBQSxDQUFBMkIsYUFBQSxHQUFBQSxhQUFBO0FBRTFDLE1BQU1ELFlBQVksQ0FBQztFQUN4QmIsY0FBY0EsQ0FBQSxFQUFHLENBQUM7RUFFbEJJLFNBQVNBLENBQUEsRUFBRyxDQUFDO0VBRWJJLGdCQUFnQkEsQ0FBQSxFQUFHLENBQUM7QUFDdEI7QUFBQ3JCLE9BQUEsQ0FBQTBCLFlBQUEsR0FBQUEsWUFBQTtBQUVNLFNBQVNMLGdCQUFnQkEsQ0FBQ0QsV0FBVyxFQUFFO0VBQzVDTyxhQUFhLENBQUNOLGdCQUFnQixDQUFDRCxXQUFXLENBQUM7QUFDN0M7QUFFTyxTQUFTSCxTQUFTQSxDQUFDSCxTQUFTLEVBQUVJLHNCQUFzQixFQUFFQyxRQUFRLEdBQUcsQ0FBQyxDQUFDLEVBQUU7RUFDMUVBLFFBQVEsQ0FBQ1osb0JBQW9CLEdBQUdvQixhQUFhLENBQUNwQixvQkFBb0I7RUFDbEVvQixhQUFhLENBQUNWLFNBQVMsQ0FBQ0gsU0FBUyxFQUFFSSxzQkFBc0IsRUFBRUMsUUFBUSxDQUFDO0FBQ3RFO0FBRU8sU0FBU0ssUUFBUUEsQ0FBQ1YsU0FBUyxFQUFFQyxLQUFLLEVBQUU7RUFDekNBLEtBQUssQ0FBQ1Isb0JBQW9CLEdBQUdvQixhQUFhLENBQUNwQixvQkFBb0I7RUFDL0RvQixhQUFhLENBQUNILFFBQVEsQ0FBQ1YsU0FBUyxFQUFFQyxLQUFLLENBQUM7QUFDMUMifQ==