"use strict";

const qs = require('querystring');
const {
  remote,
  ipcRenderer: ipc
} = require('electron');
const {
  GitProcess
} = require('dugite');
class AverageTracker {
  constructor({
    limit
  } = {
    limit: 10
  }) {
    // for now this serves a dual purpose - # of values tracked AND # discarded prior to starting avg calculation
    this.limit = limit;
    this.sum = 0;
    this.values = [];
  }
  addValue(value) {
    if (this.values.length >= this.limit) {
      const discardedValue = this.values.shift();
      this.sum -= discardedValue;
    }
    this.values.push(value);
    this.sum += value;
  }
  getAverage() {
    if (this.enoughData()) {
      return this.sum / this.limit;
    } else {
      return null;
    }
  }
  getLimit() {
    return this.limit;
  }
  enoughData() {
    return this.values.length === this.limit;
  }
}
const query = qs.parse(window.location.search.substr(1));
const sourceWebContentsId = remote.getCurrentWindow().webContents.id;
const operationCountLimit = parseInt(query.operationCountLimit, 10);
const averageTracker = new AverageTracker({
  limit: operationCountLimit
});
const childPidsById = new Map();
const destroyRenderer = () => {
  if (!managerWebContents.isDestroyed()) {
    managerWebContents.removeListener('crashed', destroyRenderer);
    managerWebContents.removeListener('destroyed', destroyRenderer);
  }
  const win = remote.BrowserWindow.fromWebContents(remote.getCurrentWebContents());
  if (win && !win.isDestroyed()) {
    win.destroy();
  }
};
const managerWebContentsId = parseInt(query.managerWebContentsId, 10);
const managerWebContents = remote.webContents.fromId(managerWebContentsId);
if (managerWebContents && !managerWebContents.isDestroyed()) {
  managerWebContents.on('crashed', destroyRenderer);
  managerWebContents.on('destroyed', destroyRenderer);
  window.onbeforeunload = () => {
    managerWebContents.removeListener('crashed', destroyRenderer);
    managerWebContents.removeListener('destroyed', destroyRenderer);
  };
}
const channelName = query.channelName;
ipc.on(channelName, (event, {
  type,
  data
}) => {
  if (type === 'git-exec') {
    const {
      args,
      workingDir,
      options,
      id
    } = data;
    if (args) {
      document.getElementById('command').textContent = `git ${args.join(' ')}`;
    }
    options.processCallback = child => {
      childPidsById.set(id, child.pid);
      child.on('error', err => {
        event.sender.sendTo(managerWebContentsId, channelName, {
          sourceWebContentsId,
          type: 'git-spawn-error',
          data: {
            id,
            err
          }
        });
      });
      child.stdin.on('error', err => {
        event.sender.sendTo(managerWebContentsId, channelName, {
          sourceWebContentsId,
          type: 'git-stdin-error',
          data: {
            id,
            stdin: options.stdin,
            err
          }
        });
      });
    };
    const spawnStart = performance.now();
    GitProcess.exec(args, workingDir, options).then(({
      stdout,
      stderr,
      exitCode
    }) => {
      const timing = {
        spawnTime: spawnEnd - spawnStart,
        execTime: performance.now() - spawnEnd
      };
      childPidsById.delete(id);
      event.sender.sendTo(managerWebContentsId, channelName, {
        sourceWebContentsId,
        type: 'git-data',
        data: {
          id,
          average: averageTracker.getAverage(),
          results: {
            stdout,
            stderr,
            exitCode,
            timing
          }
        }
      });
    }, err => {
      const timing = {
        spawnTime: spawnEnd - spawnStart,
        execTime: performance.now() - spawnEnd
      };
      childPidsById.delete(id);
      event.sender.sendTo(managerWebContentsId, channelName, {
        sourceWebContentsId,
        type: 'git-data',
        data: {
          id,
          average: averageTracker.getAverage(),
          results: {
            stdout: err.stdout,
            stderr: err.stderr,
            exitCode: err.code,
            signal: err.signal,
            timing
          }
        }
      });
    });
    const spawnEnd = performance.now();
    averageTracker.addValue(spawnEnd - spawnStart);

    // TODO: consider using this to avoid duplicate write operations upon crashing.
    // For now we won't do this to avoid clogging up ipc channel
    // event.sender.sendTo(managerWebContentsId, channelName, {sourceWebContentsId, type: 'exec-started', data: {id}});

    if (averageTracker.enoughData() && averageTracker.getAverage() > 20) {
      event.sender.sendTo(managerWebContentsId, channelName, {
        type: 'slow-spawns'
      });
    }
  } else if (type === 'git-cancel') {
    const {
      id
    } = data;
    const childPid = childPidsById.get(id);
    if (childPid !== undefined) {
      require('tree-kill')(childPid, 'SIGINT', () => {
        event.sender.sendTo(managerWebContentsId, channelName, {
          sourceWebContentsId,
          type: 'git-cancelled',
          data: {
            id,
            childPid
          }
        });
      });
      childPidsById.delete(id);
    }
  } else {
    throw new Error(`Could not identify type ${type}`);
  }
});
ipc.sendTo(managerWebContentsId, channelName, {
  sourceWebContentsId,
  type: 'renderer-ready',
  data: {
    pid: process.pid
  }
});
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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