"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
var _fsExtra = _interopRequireDefault(require("fs-extra"));
var _path = _interopRequireDefault(require("path"));
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
// Measure elapsed durations from specific beginning points.

// The maximum number of marks within a single DurationSet. A DurationSet will be automatically finished if this many
// marks are recorded.
const MAXIMUM_MARKS = 100;

// Flush all non-active DurationSets to disk each time that this many marks have been accumulated.
const PERSIST_INTERVAL = 1000;

// A sequence of durations measured from a fixed beginning point.
class DurationSet {
  constructor(name) {
    this.name = name;
    this.startTimestamp = performance.now();
    this.marks = [];
    this.markCount = 0;
    if (atom.config.get('github.performanceToConsole')) {
      // eslint-disable-next-line no-console
      console.log('%cbegin %c%s:begin', 'font-weight: bold', 'font-weight: normal; font-style: italic; color: blue', this.name);
    }
    if (atom.config.get('github.performanceToProfile')) {
      // eslint-disable-next-line no-console
      console.profile(this.name);
    }
  }
  mark(eventName) {
    const duration = performance.now() - this.startTimestamp;
    if (atom.config.get('github.performanceToConsole')) {
      // eslint-disable-next-line no-console
      console.log('%cmark %c%s:%s %c%dms', 'font-weight: bold', 'font-weight: normal; font-style: italic; color: blue', this.name, eventName, 'font-style: normal; color: black', duration);
    }
    if (atom.config.get('github.performanceToDirectory') !== '') {
      this.marks.push({
        eventName,
        duration
      });
    }
    this.markCount++;
    if (this.markCount >= MAXIMUM_MARKS) {
      this.finish();
    }
  }
  finish() {
    this.mark('finish');
    if (atom.config.get('github.performanceToProfile')) {
      // eslint-disable-next-line no-console
      console.profileEnd(this.name);
    }
  }
  serialize() {
    return {
      name: this.name,
      markers: this.marks
    };
  }
  getCount() {
    return this.marks.length;
  }
}
let durationSets = [];
let totalMarkCount = 0;
const activeSets = new Map();
function shouldCapture(seriesName, eventName) {
  const anyActive = ['Console', 'Directory', 'Profile'].some(kind => {
    const value = atom.config.get(`github.performanceTo${kind}`);
    return value !== '' && value !== false;
  });
  if (!anyActive) {
    return false;
  }
  const mask = new RegExp(atom.config.get('github.performanceMask'));
  if (!mask.test(`${seriesName}:${eventName}`)) {
    return false;
  }
  return true;
}
const yardstick = {
  async save() {
    const destDir = atom.config.get('github.performanceToDirectory');
    if (destDir === '' || destDir === undefined || destDir === null) {
      return;
    }
    const fileName = _path.default.join(destDir, `performance-${Date.now()}.json`);
    await new Promise((resolve, reject) => {
      _fsExtra.default.ensureDir(destDir, err => err ? reject(err) : resolve());
    });
    const payload = JSON.stringify(durationSets.map(set => set.serialize()));
    await _fsExtra.default.writeFile(fileName, payload, {
      encoding: 'utf8'
    });
    if (atom.config.get('github.performanceToConsole')) {
      // eslint-disable-next-line no-console
      console.log('%csaved %c%d series to %s', 'font-weight: bold', 'font-weight: normal; color: black', durationSets.length, fileName);
    }
    durationSets = [];
  },
  begin(seriesName) {
    if (!shouldCapture(seriesName, 'begin')) {
      return;
    }
    const ds = new DurationSet(seriesName);
    activeSets.set(seriesName, ds);
  },
  mark(seriesName, eventName) {
    if (seriesName instanceof Array) {
      for (let i = 0; i < seriesName.length; i++) {
        this.mark(seriesName[i], eventName);
      }
      return;
    }
    if (!shouldCapture(seriesName, eventName)) {
      return;
    }
    const ds = activeSets.get(seriesName);
    if (ds === undefined) {
      return;
    }
    ds.mark(eventName);
  },
  finish(seriesName) {
    if (seriesName instanceof Array) {
      for (let i = 0; i < seriesName.length; i++) {
        this.finish(seriesName[i]);
      }
      return;
    }
    if (!shouldCapture(seriesName, 'finish')) {
      return;
    }
    const ds = activeSets.get(seriesName);
    if (ds === undefined) {
      return;
    }
    ds.finish();
    durationSets.push(ds);
    activeSets.delete(seriesName);
    totalMarkCount += ds.getCount();
    if (totalMarkCount >= PERSIST_INTERVAL) {
      totalMarkCount = 0;
      this.save();
    }
  },
  async flush() {
    durationSets.push(...activeSets.values());
    activeSets.clear();
    await this.save();
  }
};
var _default = yardstick;
exports.default = _default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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